import ReactDOM from "react-dom";
import { Toaster, toast } from "react-hot-toast";
import React, { useEffect, useState } from "react";
import FilterBy from "./FilterBy";
import { Container } from "react-bootstrap";

import "react-progress-2/main.css";
import Progress from "react-progress-2";

const CategoryFilter = (props) => {
    const categoryId = props.slug;
    const [category, setCategory] = useState([]);

    const [categories, setCategories] = useState([]);
    const [brands, setBrands] = useState([]);
    const [products, setProducts] = useState([]);
    const [totalProducts, setTotalProducts] = useState(0);
    const [perPages, setPerPages] = useState(0);
    const [currentPage, setCurrentPage] = useState(0);
    const [loading, setLoading] = useState(false);

    const [sort, setSort] = useState("desc");

    const [show, setShow] = useState(false);

    const handleClose = () => setShow(false);
    const handleShow = () => setShow(true);

    const loadData = async () => {
        setLoading(true);
        await axios
            .get(`/api/categories`)
            .then((response) => {
                if (response.data.result === "success") {
                    setCategory(response.data.category);
                    setProducts(response.data.products);
                    setTotalProducts(response.data.totalProducts);
                    setPerPages(response.data.perPages);
                    setCurrentPage(response.data.currentPage);
                    setSort(response.data.sort);
                }
            })
            .catch((error) => {
                toast.error(error.message);
            });

        await axios
            .post(`/api/categories/filter`, {
                category: categoryId,
            })
            .then((response) => {
                if (response.data.result === "success") {
                    setCategories(response.data.categories);
                    setBrands(response.data.brands);
                }
            })
            .catch((error) => {
                toast.error(error.message);
            });
        setLoading(false);
    };

    useEffect(() => {
        loadData();
    }, []);

    const [selectedCuisines, setSelectedCuisines] = useState([]);
    const [selectedCategories, setSelectedCategories] = useState([]);

    const [selectedBrands, setSelectedBrands] = useState([]);
    const [checkedBrands, setCheckedBrands] = useState([]);

    async function getData(pageNumber = 1) {
        window.scrollTo(0, 0);
        Progress.show();
        setLoading(true);

        try {
            const filters = [];
            // Brand
            const brandFilters = [];
            if (selectedBrands.length) {
                brandFilters.push(`brands=${selectedBrands.join(",")}`);
            }

            // Size
            if (selectedCuisines.length) {
                filters.push(`categories=${selectedCuisines.join(",")}`);
            }
            const filterQueryString =
                filters.length > 0 ? `&${filters.join("&")}` : "";

            const filterQuerySizes =
                filters.length > 0 ? `&${filters.join("&")}` : "";

            const filterQueryBrands =
                filters.length > 0 ? `&${filters.join("&")}` : "";
            const url = `/api/categories?${filterQueryString}&sizes=${filterQuerySizes}&brands=${filterQueryBrands}&page=${pageNumber}&sort=${sort}`;

            const response = await axios.get(url);

            setProducts(response.data.products);
            setTotalProducts(response.data.totalProducts);
            setPerPages(response.data.itemsCountPerPage);
            setCurrentPage(response.data.currentPage);

            Progress.hide();
            setLoading(false);
        } catch (error) {
            toast.error(error.message);
            Progress.hide();
            setLoading(false);
        }
    }

    const handleChangeCategory = (id, type) => {
        const updatedCategories = categories.map((item) => {
            if (type === "category") {
                return item.id === id
                    ? { ...item, checked: !item.checked }
                    : item;
            } else if (type === "subCategory") {
                const updatedSubCategory = item.subCategory.map(
                    (subCategory) => {
                        return subCategory.id === id
                            ? { ...subCategory, checked: !subCategory.checked }
                            : subCategory;
                    }
                );
                return { ...item, subCategory: updatedSubCategory };
            } else if (type === "childCategory") {
                const updatedSubCategory = item.subCategory.map(
                    (subCategory) => {
                        const updatedChildCategory =
                            subCategory.childCategory.map((childCategory) => {
                                return childCategory.id === id
                                    ? {
                                          ...childCategory,
                                          checked: !childCategory.checked,
                                      }
                                    : childCategory;
                            });
                        return {
                            ...subCategory,
                            childCategory: updatedChildCategory,
                        };
                    }
                );
                return { ...item, subCategory: updatedSubCategory };
            }
            return item;
        });

        setCategories(updatedCategories);
    };

    const handleChangeBrand = (id, type) => {
        const updateBrands = brands.map((item) => {
            if (type === "brand") {
                return item.id === id
                    ? { ...item, checked: !item.checked }
                    : item;
            }
            return item;
        });

        setBrands(updateBrands);
    };

    const changePrice = async () => {
        const filters = [];
        const brandFilters = [];

        const cuisinesChecked = categories
            .filter((item) => item.checked)
            .map((item) => ({
                id: item.id,
                title: item.title,
                type: "category",
            }));

        const subCategoriesChecked = categories
            .flatMap((item) =>
                item.subCategory.filter((subItem) => subItem.checked)
            )
            .map((subItem) => ({
                id: subItem.id,
                title: subItem.title,
                type: "subCategory",
            }));

        const childCategoriesChecked = categories
            .flatMap((item) =>
                item.subCategory.flatMap((subItem) =>
                    subItem.childCategory.filter(
                        (childItem) => childItem.checked
                    )
                )
            )
            .map((childItem) => ({
                id: childItem.id,
                title: childItem.title,
                type: "childCategory",
            }));

        const brandsChecked = brands
            .filter((item) => item.checked)
            .map((item) => ({
                id: item.id,
                title: item.title,
                type: "brand",
            }));

        const allCheckedIds = [
            ...cuisinesChecked,
            ...subCategoriesChecked,
            ...childCategoriesChecked,
        ];

        const selectedCuisinesIds = allCheckedIds.map((item) => item.id);
        setSelectedCuisines(selectedCuisinesIds);
        setSelectedCategories(allCheckedIds);

        // Brand
        const selectedBrandIds = brandsChecked.map((item) => item.id);
        setSelectedBrands(brandsChecked);
        setCheckedBrands(brandsChecked);

        if (allCheckedIds.length) {
            filters.push(`categories=${selectedCuisinesIds.join(",")}`);
        }

        if (selectedBrandIds.length) {
            brandFilters.push(`brands=${selectedBrandIds.join(",")}`);
        }
        const apiUrl = `/api/categories?${
            filters.length > 0 ? filters.join("&") : ""
        }${brandFilters.length > 0 ? `&` + brandFilters.join("&") : ""}${
            sort ? `&sort=${sort}` : ""
        }`;

        await axios
            .get(apiUrl)
            .then((response) => {
                setProducts(response.data.products);
                setTotalProducts(response.data.totalProducts);
                setPerPages(response.data.itemsCountPerPage);
                setCurrentPage(response.data.currentPage);
            })
            .catch((error) => {
                toast.error(error.message);
            });
    };

    useEffect(() => {
        changePrice();
    }, [totalProducts, categories, sort, brands]);

    return (
        <>
            <Progress.Component
                style={{ background: "#99999978", height: "5px" }}
                thumbStyle={{
                    background: "#f29f05",
                    height: "5px",
                }}
            />
            <Toaster />
            <section className="product section-gap">
                <Container>
                    <FilterBy
                        products={products}
                        category={category}
                        totalProducts={totalProducts}
                        loading={loading}
                        brands={brands}
                        handleChangeBrand={handleChangeBrand}
                        checkedBrands={checkedBrands}
                        categories={categories}
                        selectedCategories={selectedCategories}
                        handleChange={handleChangeCategory}
                        sort={sort}
                        setSort={setSort}
                        perPages={perPages}
                        currentPage={currentPage}
                        getData={getData}
                        show={show}
                        handleClose={handleClose}
                        handleShow={handleShow}
                    />
                </Container>
            </section>
        </>
    );
};

export default CategoryFilter;

if (document.getElementById("categoryFilter")) {
    ReactDOM.render(
        <CategoryFilter {...categoryFilter.dataset} />,
        document.getElementById("categoryFilter")
    );
}
